import os
import yaml
import numpy as np
import argparse


def load_kernel_data(config: dict, config_path: str | None = None) -> None:
    """
    Load the kernel generator field K_mu(x) from the diagnostics repo outputs.

    Parameters
    ----------
    config : dict
        The configuration mapping parsed from YAML. Must contain a
        ``kernel_path`` entry pointing to the kernel file.
    config_path : str, optional
        Absolute path of the configuration file. If provided, relative
        ``kernel_path`` and ``data_dir`` values are resolved relative to
        this directory.  If omitted, relative paths are interpreted
        relative to the current working directory.

    Saves
    -----
    kernel.npy : NumPy file
        A copy of the loaded kernel saved into ``data_dir`` as
        ``kernel.npy``.  Ensures that downstream steps can locate the
        kernel at a stable location.
    """
    kernel_path_cfg = config.get('kernel_path', None)
    if kernel_path_cfg is None:
        raise ValueError('kernel_path not specified in config.yaml')

    # Determine base directory for relative resolution
    if config_path is not None:
        base_dir = os.path.dirname(os.path.abspath(config_path))
    else:
        base_dir = os.getcwd()

    # Resolve kernel path relative to config if necessary
    if os.path.isabs(kernel_path_cfg):
        kernel_path = kernel_path_cfg
    else:
        kernel_path = os.path.normpath(os.path.join(base_dir, kernel_path_cfg))

    # Load kernel array; support .npy or CSV
    if kernel_path.endswith('.npy'):
        kernel = np.load(kernel_path, allow_pickle=True)
    else:
        kernel = np.loadtxt(kernel_path, delimiter=',')

    # Resolve data_dir relative to config if necessary
    data_dir_cfg = config.get('data_dir', 'data')
    if os.path.isabs(data_dir_cfg):
        data_dir = data_dir_cfg
    else:
        data_dir = os.path.join(base_dir, data_dir_cfg)
    os.makedirs(data_dir, exist_ok=True)

    out_path = os.path.join(data_dir, 'kernel.npy')
    np.save(out_path, kernel)
    print(f'Loaded kernel from {kernel_path}, saved to {out_path}')


# Alias for discrete-gauge import
load_kernel = load_kernel_data


def main(config_path: str) -> None:
    """CLI entry point for kernel loading."""
    with open(config_path) as f:
        cfg = yaml.safe_load(f)
    load_kernel_data(cfg, config_path=config_path)


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Load kernel diagnostics data')
    parser.add_argument('--config', default='config.yaml', help='Path to config file')
    args = parser.parse_args()
    main(args.config)
